"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApolloBaseDriver = void 0;
const server_1 = require("@apollo/server");
const server_plugin_landing_page_graphql_playground_1 = require("@apollo/server-plugin-landing-page-graphql-playground");
const errors_1 = require("@apollo/server/errors");
const express4_1 = require("@apollo/server/express4");
const disabled_1 = require("@apollo/server/plugin/disabled");
const drainHttpServer_1 = require("@apollo/server/plugin/drainHttpServer");
const common_1 = require("@nestjs/common");
const load_package_util_1 = require("@nestjs/common/utils/load-package.util");
const shared_utils_1 = require("@nestjs/common/utils/shared.utils");
const graphql_1 = require("@nestjs/graphql");
const graphql_2 = require("graphql");
const omit = require("lodash.omit");
const async_iterator_util_1 = require("../utils/async-iterator.util");
const apolloPredefinedExceptions = {
    [common_1.HttpStatus.BAD_REQUEST]: errors_1.ApolloServerErrorCode.BAD_REQUEST,
    [common_1.HttpStatus.UNPROCESSABLE_ENTITY]: errors_1.ApolloServerErrorCode.BAD_USER_INPUT,
    [common_1.HttpStatus.UNAUTHORIZED]: 'UNAUTHENTICATED',
    [common_1.HttpStatus.FORBIDDEN]: 'FORBIDDEN',
};
class ApolloBaseDriver extends graphql_1.AbstractGraphQLDriver {
    get instance() {
        return this.apolloServer;
    }
    async start(apolloOptions) {
        const httpAdapter = this.httpAdapterHost.httpAdapter;
        const platformName = httpAdapter.getType();
        if (platformName === 'express') {
            await this.registerExpress(apolloOptions);
        }
        else if (platformName === 'fastify') {
            await this.registerFastify(apolloOptions);
        }
        else {
            throw new Error(`No support for current HttpAdapter: ${platformName}`);
        }
    }
    stop() {
        return this.apolloServer?.stop();
    }
    async mergeDefaultOptions(options) {
        let defaults = {
            path: '/graphql',
            fieldResolverEnhancers: [],
            stopOnTerminationSignals: false,
        };
        if ((options.playground === undefined &&
            process.env.NODE_ENV !== 'production') ||
            options.playground) {
            const playgroundOptions = typeof options.playground === 'object' ? options.playground : undefined;
            defaults = {
                ...defaults,
                plugins: [
                    (0, server_plugin_landing_page_graphql_playground_1.ApolloServerPluginLandingPageGraphQLPlayground)(playgroundOptions),
                ],
            };
        }
        else if ((options.playground === undefined &&
            process.env.NODE_ENV === 'production') ||
            options.playground === false) {
            defaults = {
                ...defaults,
                plugins: [(0, disabled_1.ApolloServerPluginLandingPageDisabled)()],
            };
        }
        options = await super.mergeDefaultOptions(options, omit(defaults, 'plugins'));
        options.plugins = (options.plugins || []).concat(defaults.plugins || []);
        this.wrapContextResolver(options);
        this.wrapFormatErrorFn(options);
        return options;
    }
    subscriptionWithFilter(instanceRef, filterFn, createSubscribeContext) {
        return (...args) => (0, async_iterator_util_1.createAsyncIterator)(createSubscribeContext()(...args), (payload) => filterFn.call(instanceRef, payload, ...args.slice(1)));
    }
    async registerExpress(options, { preStartHook } = {}) {
        const { path, typeDefs, resolvers, schema } = options;
        const httpAdapter = this.httpAdapterHost.httpAdapter;
        // Workaround: GraphQL playground requires body to be present
        // otherwise, it shows the "req.body is not set; this probably means you forgot to set up the json middleware before the Apollo Server middleware." error.
        // The latest version of "body-parser" does not set the body if there is no payload.
        // @see https://github.com/nestjs/graphql/issues/3451
        // @see https://github.com/nestjs/nest/pull/14443
        httpAdapter.use(path, (req, _, next) => {
            req.body = req.body ?? {};
            next();
        });
        const app = httpAdapter.getInstance();
        const drainHttpServerPlugin = (0, drainHttpServer_1.ApolloServerPluginDrainHttpServer)({
            httpServer: httpAdapter.getHttpServer(),
        });
        preStartHook?.();
        const server = new server_1.ApolloServer({
            typeDefs,
            resolvers,
            schema,
            ...options,
            plugins: options.plugins
                ? options.plugins.concat([drainHttpServerPlugin])
                : [drainHttpServerPlugin],
        });
        await server.start();
        app.use(path, (0, express4_1.expressMiddleware)(server, {
            context: options.context,
        }));
        this.apolloServer = server;
    }
    async registerFastify(options, { preStartHook } = {}) {
        const { fastifyApolloDrainPlugin, fastifyApolloHandler } = (0, load_package_util_1.loadPackage)('@as-integrations/fastify', 'GraphQLModule', () => require('@as-integrations/fastify'));
        const httpAdapter = this.httpAdapterHost.httpAdapter;
        const app = httpAdapter.getInstance();
        const { path, typeDefs, resolvers, schema } = options;
        const apolloDrainPlugin = fastifyApolloDrainPlugin(app);
        preStartHook?.();
        const server = new server_1.ApolloServer({
            typeDefs,
            resolvers,
            schema,
            ...options,
            plugins: options.plugins
                ? options.plugins.concat([apolloDrainPlugin])
                : [apolloDrainPlugin],
        });
        await server.start();
        app.route({
            url: path,
            method: ['GET', 'POST', 'OPTIONS'],
            handler: fastifyApolloHandler(server, {
                context: options.context,
            }),
        });
        this.apolloServer = server;
    }
    wrapFormatErrorFn(options) {
        if (options.autoTransformHttpErrors === false) {
            return;
        }
        if (options.formatError) {
            const origFormatError = options.formatError;
            const transformHttpErrorFn = this.createTransformHttpErrorFn();
            options.formatError = (formattedError, err) => {
                formattedError = transformHttpErrorFn(formattedError, err);
                return origFormatError(formattedError, err);
            };
        }
        else {
            options.formatError =
                this.createTransformHttpErrorFn();
        }
    }
    createTransformHttpErrorFn() {
        return (formattedError, originalError) => {
            const exceptionRef = (0, errors_1.unwrapResolverError)(originalError);
            const isHttpException = exceptionRef?.response?.statusCode && exceptionRef?.status;
            if (!isHttpException) {
                return formattedError;
            }
            let error;
            const httpStatus = exceptionRef?.status;
            if (httpStatus in apolloPredefinedExceptions) {
                error = new graphql_2.GraphQLError(exceptionRef?.message, {
                    path: formattedError.path,
                    extensions: {
                        ...formattedError.extensions,
                        code: apolloPredefinedExceptions[httpStatus],
                    },
                });
            }
            else {
                error = new graphql_2.GraphQLError(exceptionRef.message, {
                    path: formattedError.path,
                    extensions: {
                        ...formattedError.extensions,
                        code: errors_1.ApolloServerErrorCode.INTERNAL_SERVER_ERROR,
                        status: httpStatus,
                    },
                });
            }
            if (exceptionRef?.response) {
                error.extensions['originalError'] = exceptionRef.response;
            }
            error.locations = formattedError.locations;
            return error;
        };
    }
    wrapContextResolver(targetOptions, originalOptions = { ...targetOptions }) {
        if (!targetOptions.context) {
            targetOptions.context = async (contextOrRequest) => {
                return {
                    // New ApolloServer fastify integration has Request as first parameter to the Context function
                    req: contextOrRequest.req ?? contextOrRequest,
                };
            };
        }
        else if ((0, shared_utils_1.isFunction)(targetOptions.context)) {
            targetOptions.context = async (...args) => {
                const ctx = await originalOptions.context(...args);
                const contextOrRequest = args[0];
                return this.assignReqProperty(ctx, contextOrRequest.req ?? contextOrRequest);
            };
        }
        else {
            targetOptions.context = async (contextOrRequest) => {
                return this.assignReqProperty(originalOptions.context, contextOrRequest.req ?? contextOrRequest);
            };
        }
    }
    assignReqProperty(ctx, req) {
        if (!ctx) {
            return { req };
        }
        if (typeof ctx !== 'object' ||
            (ctx && ctx.req && typeof ctx.req === 'object')) {
            return ctx;
        }
        ctx.req = req;
        return ctx;
    }
}
exports.ApolloBaseDriver = ApolloBaseDriver;
