"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFieldsAndDecoratorForType = getFieldsAndDecoratorForType;
require("reflect-metadata");
const decorators_1 = require("../../decorators");
const class_type_enum_1 = require("../../enums/class-type.enum");
const graphql_constants_1 = require("../../graphql.constants");
const unable_to_find_fields_error_1 = require("../errors/unable-to-find-fields.error");
const lazy_metadata_storage_1 = require("../storages/lazy-metadata.storage");
const type_metadata_storage_1 = require("../storages/type-metadata.storage");
function getFieldsAndDecoratorForType(objType, options) {
    const classType = Reflect.getMetadata(graphql_constants_1.CLASS_TYPE_METADATA, objType);
    if (!classType) {
        throw new unable_to_find_fields_error_1.UnableToFindFieldsError(objType.name);
    }
    lazy_metadata_storage_1.LazyMetadataStorage.load([objType], {
        skipFieldLazyMetadata: true,
    });
    const [classMetadata, decoratorFactory] = getClassMetadataAndFactoryByTargetAndType(classType, objType);
    type_metadata_storage_1.TypeMetadataStorage.loadClassPluginMetadata([classMetadata]);
    type_metadata_storage_1.TypeMetadataStorage.compileClassMetadata([classMetadata], options);
    let fields = classMetadata?.properties;
    if (!fields) {
        throw new unable_to_find_fields_error_1.UnableToFindFieldsError(objType.name);
    }
    fields = inheritClassFields(objType, fields, options);
    return {
        fields,
        decoratorFactory,
    };
}
function getClassMetadataAndFactoryByTargetAndType(classType, objType) {
    switch (classType) {
        case class_type_enum_1.ClassType.ARGS:
            return [
                type_metadata_storage_1.TypeMetadataStorage.getArgumentsMetadataByTarget(objType),
                decorators_1.ArgsType,
            ];
        case class_type_enum_1.ClassType.OBJECT:
            return [
                type_metadata_storage_1.TypeMetadataStorage.getObjectTypeMetadataByTarget(objType),
                decorators_1.ObjectType,
            ];
        case class_type_enum_1.ClassType.INPUT:
            return [
                type_metadata_storage_1.TypeMetadataStorage.getInputTypeMetadataByTarget(objType),
                decorators_1.InputType,
            ];
        case class_type_enum_1.ClassType.INTERFACE:
            return [
                type_metadata_storage_1.TypeMetadataStorage.getInterfaceMetadataByTarget(objType),
                decorators_1.InterfaceType,
            ];
    }
}
function inheritClassFields(objType, fields, options) {
    try {
        const parentClass = Object.getPrototypeOf(objType);
        if (parentClass === Function) {
            return fields;
        }
        const { fields: parentFields } = getFieldsAndDecoratorForType(parentClass, options);
        return inheritClassFields(parentClass, [...parentFields, ...fields], options);
    }
    catch {
        return fields;
    }
}
